<?php

namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Vendor\Manager;
use Exception;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Foundation\Auth\RegistersUsers;
use App\Notifications\ManagerNotification;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;

class ManagerController extends Controller
{

    /*
    |--------------------------------------------------------------------------
    | Register
    |--------------------------------------------------------------------------
    |
    | This  handles the registration of new manager as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;


    /**
     * Method for show Manager page
     * @return view
     */
    public function index()
    {
        $page_title     = "Manager List";
        $data = Manager::auth()->with('parlourlist')->orderBYDESC('id')->paginate(6);
        return view('vendor.sections.manager.index', compact('data', 'page_title'));
    }

    /**
     * Method for show doctor-care create page
     * @param string $slug
     * @param \Illuminate\Http\Request  $request
     */
    public function showForm()
    {
        $page_title      = "Manager Create";
        $vendor_id = auth()->user()->id;

        return view('vendor.sections.manager.create', compact(
            'page_title',
            'vendor_id'
        ));
    }

    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {

        $validated = $this->validator($request->all())->validate();
        $validated['password']                     = Hash::make($validated['password']);
        $validated['username']                     = make_username(Str::slug($validated['firstname']), Str::slug($validated['lastname']), "managers");
        $validated['vendor_id']                    = $request->vendor_id;


        if($request->hasFile("image")) {
            $image = upload_file($validated['image'],'manager-profile');
            $upload_image = upload_files_from_path_dynamic([$image['dev_path']],'manager-profile');
            delete_file($image['dev_path']);
            $validated['image']     = $upload_image;
        }


        event(new Registered($this->create($validated)));
        try {
            Notification::route('mail', $validated['email'])->notify(new ManagerNotification($data = $validated, $request->password));
        } catch (Exception) {
        }
        return $this->registered($request);
    }


    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    public function validator(array $data)
    {

        return Validator::make($data, [
            'firstname'     => 'required|string|max:60',
            'lastname'      => 'required|string|max:60',
            'password'      => 'required|string|min:6',
            'email'         => 'required|string|email|max:150|unique:managers,email',
            'image'         => 'required',
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        return Manager::create($data);
    }


    /**
     * The user has been registered.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  mixed  $user
     * @return mixed
     */
    protected function registered(Request $request)
    {
        return redirect()->route('vendors.manager.index')->with(['success' => ["Manager Created Successfully!"]]);
    }


    /**
     * Method for delete the Manager information
     * @param \Illuminate\Http\Request $request
     */
    public function delete(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'target'    => 'required|numeric',
        ]);


        $data   = Manager::find($request->target);

        try {
            $data->delete();
        } catch (Exception $e) {
            return back()->with(['error'    =>  ['Something went wrong. Please try again!']]);
        }
        return back()->with(['success'  =>  ['Manager deleted successfully.']]);
    }

    /**
     * Method for update manager
     * @param string
     * @param \Illuminate\Http\Request $request
     */
    public function update(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'target'              => 'required|numeric|exists:managers,id',
            'first_name_edit'     => 'required|string',
            'last_name_edit'      => 'required|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput()->with("modal", "manager-update");
        }

        $validated = $validator->validate();

        $manager = Manager::find($request->target);

        try {
            $manager->update([
                'firstname' => $validated['first_name_edit'],
                'lastname'  => $validated['last_name_edit'],
            ]);
        } catch (Exception $e) {
            return back()->with(['error' => ['Something went wrong! Please try again']]);
        }

        return back()->with(['success' => ['Manager updated successfully!']]);
    }


}
