<?php

namespace App\Http\Controllers\Vendor;

use App\Constants\GlobalConst;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Admin\ParlourList;
use App\Models\Admin\ParlourHasService;
use Illuminate\Support\Facades\Validator;
use App\Models\Admin\ParlourListHasSchedule;
use Illuminate\Validation\ValidationException;
use App\Models\Admin\Area;
use App\Models\Vendor\Manager;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;
use Exception;
use App\Http\Helpers\Response;
use App\Models\Vendor\ParlourHasStuff;
use App\Models\Admin\BasicSettings;
use App\Models\ParlourBooking;
use App\Models\Vendor\ParlourImage;

class ParlourListController extends Controller
{
    /**
     * Method for show parlour list page
     * @return view
     */
    public function index()
    {
        $page_title     = "Salon List";
        $vendor_id      = auth()->user()->id;

        $parlour_lists  = ParlourList::with('parlourImage')->where('vendor_id', $vendor_id)->orderBYDESC('id')->paginate(3);

        return view('vendor.sections.parlour-list.index', compact(
            'page_title',
            'parlour_lists',
         

        ));
    }
    /**
     * Method for show doctor-care create page
     * @param string $slug
     * @param \Illuminate\Http\Request  $request
     */
    public function create()
    {
        $page_title      = "Salon Create";
        $areas           = Area::where('status', true)->get();
        $manager         = Manager::where('vendor_id', auth()->id())->where('status', true)->where('assign_status', false)->get();
        $todayDate       = Carbon::now()->format('d F, Y');
        $vendor_id       = auth()->user()->id;
        $parlour_has_stuff = ParlourHasStuff::where('vendor_id', $vendor_id)->where('status', true)->get();
        return view('vendor.sections.parlour-list.create', compact(
            'page_title',
            'areas',
            'todayDate',
            'vendor_id',
            'manager',
            'parlour_has_stuff',
        ));
    }
    public function getScheduleDays()
    {

        return view('vendor.components.parlour-list.schedule-item');
    }
    public function getService()
    {

        return view('vendor.components.parlour-list.service-item');
    }

    public function getStuff()
    {
        return view('vendor.components.parlour-list.stuff-item');
    }
    /**
     * Method for store parlour list information
     * @param \Illuminate\Http\Request $request
     */
    public function store(Request $request)
    {


        $validator = Validator::make($request->all(), [
            'vendor_id'        => 'required|integer',
            'area'             => 'required|integer',
            'name'             => 'required|string|max:50',
            'manager'          => 'required|integer',
            'experience'       => 'required|string|max:100',
            'speciality'       => 'nullable',
            'contact'          => 'required',
            'address'          => 'nullable',
            'off_days'         => 'required|array',
            'number_of_dates'  => 'required|integer',
            'service_name'     => 'required|array',
            'service_name.*'   => 'required|string',
            'stuff_name'       => 'required|array',
            'stuff_name.*'     => 'required|string',
            'price'            => 'required|array',
            'price.*'          => 'required|string',
            'from_time'        => 'required|array',
            'from_time.*'      => 'required|string',
            'to_time'          => 'required|array',
            'to_time.*'        => 'required|string|after:from_time.*',
            'max_client'       => 'required|array',
            'max_client.*'     => 'required|integer',
            'image'            => 'nullable',
            'preview'            => 'nullable',
            'preview.*'            => 'nullable',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput($request->all());
        }

        $basic_setting = BasicSettings::first();

        $validated          = $validator->validate();

        $validated['slug']  = Str::uuid();
        $validated['area_id']        = $validated['area'];
        $validated['manager_id']     = $validated['manager'];



        $offDaysArray = $request->input('off_days', []);

        $offDaysString = implode(',', $offDaysArray);

        $validated['off_days']     = $offDaysString;

        $managerId =  $validated['manager_id'];


        $validated['active_status']  = GlobalConst::STATUSUNFREEZE;
        $validated['status']          = GlobalConst::STATUSPENDING;
      

        $service_name   = $validated['service_name'];
        $stuff_name     = $validated['stuff_name'];
        $price          = $validated['price'];
        $from_time      = $validated['from_time'];
        $to_time        = $validated['to_time'];
        $max_client     = $validated['max_client'];
        $validated      = Arr::except($validated, ['service_name', 'price', 'from_time', 'to_time', 'max_client', 'area', 'stuff_name']);
        if ($request->hasFile("image")) {
            $validated['image'] = $this->imageValidate($request, "image", null);
        }
        if ($basic_setting->min_stuff) {

            if (count($stuff_name) < $basic_setting->min_stuff) {
                return back()->with(['error' => ['Minimum Number of Stuff is  ' . $basic_setting->min_stuff . '.']]);
            }
        }
        try {
            $parlour_list = ParlourList::create($validated);

            if ($request->hasFile('preview')) {
                $images = $request->file('preview');


                foreach ($images as $image) {
                    $imageName = Str::uuid() . '.' . $image->getClientOriginalExtension();
                    $image->move(public_path('frontend/images/site-section'), $imageName);

                    ParlourImage::create([
                        'parlour_lists_id' => $parlour_list->id,
                        'image_path' => $imageName,
                    ]);
                }
            }



            if (count($from_time) > 0) {
                $days_shedule = [];
                foreach ($from_time as $key => $day_id) {
                    $days_shedule[] = [
                        'parlour_list_id'   => $parlour_list->id,
                        'from_time'         => $from_time[$key],
                        'to_time'           => $to_time[$key],
                        'max_client'        => $max_client[$key],
                        'created_at'        => now(),
                    ];
                }
                ParlourListHasSchedule::insert($days_shedule);
            }
            if (count($service_name) > 0) {
                $services = [];
                foreach ($service_name as $key => $day_id) {
                    $services[] = [
                        'parlour_list_id'   => $parlour_list->id,
                        'service_name'      => $service_name[$key],
                        'price'             => $price[$key],
                        'created_at'        => now(),
                    ];
                }
                ParlourHasService::insert($services);
            }


            if (count($stuff_name) > 0) {
                foreach ($stuff_name as $key => $stuff_id) {
                    ParlourHasStuff::where('id', $stuff_id)->update([
                        'parlour_list_id' => $parlour_list->id,
                        'status'          => GlobalConst::STATUSASSIGN,
                        'updated_at'      => now(),
                    ]);
                }
            }
            Manager::where('id', $managerId)->update([
                'assign_status' => true,
            ]);
        } catch (Exception $e) {

            return back()->with(['error' => ["Something went wrong.Please try again."]]);
        }
        return redirect()->route('vendors.parlour.list.index')->with(['success' => ["Salon Created Successfully!"]]);
    }
    /**
     * Method for show the edit parlour list page
     * @param $slug
     * @param \Illuminate\Http\Request $request
     */
    public function edit($slug)
    {

        $vendor_id              = auth()->user()->id;
        $page_title             = "Salon Edit";
        $parlour_list           = ParlourList::with('parlourImage')->where('slug', $slug)->first();
        if (!$parlour_list) return back()->with(['error' =>  ['Salon List Not Exists!']]);
        $areas                  = Area::where('status', true)->get();
        $manager                = Manager::where('vendor_id', auth()->id())->where('status', true)->get();
        $parlour_has_schedule   = ParlourListHasSchedule::where('parlour_list_id', $parlour_list->id)->get();
        $parlour_has_service    = ParlourHasService::where('parlour_list_id', $parlour_list->id)->get();
        $parlour_has_stuff      = ParlourHasStuff::where('vendor_id', $vendor_id)->where('parlour_list_id', $parlour_list->id)->get();

        return view('vendor.sections.parlour-list.edit', compact(
            'page_title',
            'parlour_list',
            'areas',
            'parlour_has_schedule',
            'parlour_has_service',
            'parlour_has_stuff',
            'vendor_id',
            'manager'
        ));
    }
    /**
     * Method for update the parlour list information
     * @param $slug
     * @param \Illuminate\Http\Request $request
     */
    public function update(Request $request, $slug)
    {

        $parlour_list = ParlourList::where('slug', $slug)->first();
        $validator = Validator::make($request->all(), [
            'area'             => 'required',
            'name'             => 'required|string|max:50',
            'manager'          => 'required|integer',
            'experience'       => 'required|string|max:100',
            'speciality'       => 'nullable',
            'contact'          => 'required',
            'address'          => 'nullable',
            'off_days'         => 'required|array',
            'number_of_dates'  => 'required|integer',
            'stuff_name'       => 'nullable|array',
            'stuff_name.*'     => 'nullable',
            'service_name'     => 'required|array',
            'service_name.*'   => 'required|string',
            'price'            => 'required|array',
            'price.*'          => 'required|string',
            'from_time'        => 'required|array',
            'from_time.*'      => 'required|string',
            'to_time'          => 'required|array',
            'to_time.*'        => 'required|string',
            'max_client'       => 'required|array',
            'max_client.*'     => 'required|integer',
            'image'            => 'nullable',
            'preview'            => 'nullable',
            'preview.*'            => 'nullable',
        ]);
        $basic_setting = BasicSettings::first();

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput($request->all());
        }

        $validated = $validator->validate();

        $validated['area_id'] = $validated['area'];
        $validated['manager_id'] = $validated['manager'];

        $offDaysArray = $request->input('off_days', []);

        $offDaysString = implode(',', $offDaysArray);

        $validated['off_days']     = $offDaysString;

        // Check if manager is changed
        if ($parlour_list->manager_id != $validated['manager_id']) {
            // Update previous manager's assign status to 0
            Manager::where('id', $parlour_list->manager_id)->update([
                'assign_status' => 0,
            ]);

            // Update new manager's assign status to 1
            Manager::where('id', $validated['manager_id'])->update([
                'assign_status' => GlobalConst::MANAGERASSIGN,
            ]);
        }

        if (ParlourList::where('contact', $validated['contact'])->where('id', '!=', $parlour_list->id)->exists()) {
            throw ValidationException::withMessages([
                'name' => "Salon already exists!",
            ]);
        }

        if ($request->hasFile('image')) {
            $validated['image'] = $this->imageValidate($request, "image", null);
        }

        $service_name = $validated['service_name'];
        $stuff_name = $validated['stuff_name'] ?? null;

        if ($basic_setting->min_stuff) {
            if (!is_array($stuff_name)) {
                $stuff_name = [];
            }

            if (count($stuff_name) < $basic_setting->min_stuff) {
                return back()->with(['error' => ['Minimum Number of Stuff is ' . $basic_setting->min_stuff . '.']]);
            }
        }

        $price = $validated['price'];
        $from_time = $validated['from_time'];
        $to_time = $validated['to_time'];
        $max_client = $validated['max_client'];
        $validated = Arr::except($validated, ['service_name', 'stuff_name', 'price', 'from_time', 'to_time', 'max_client', 'area']);

        try {
            $parlour_list->update($validated);

              if ($request->hasFile('preview')) {
                $images = $request->file('preview');


                foreach ($images as $image) {
                    $imageName = Str::uuid() . '.' . $image->getClientOriginalExtension();
                    $image->move(public_path('frontend/images/site-section'), $imageName);

                    ParlourImage::create([
                        'parlour_lists_id' => $parlour_list->id,
                        'image_path' => $imageName,
                    ]);
                }
            }


            $existing_schedules = $parlour_list->schedules()->pluck('id')->all();
            $processed_schedule_ids = [];

            foreach ($from_time as $key => $value) {
                $scheduleData = [
                    'from_time' => $from_time[$key],
                    'to_time' => $to_time[$key],
                    'max_client' => $max_client[$key],
                    'updated_at' => now(),
                ];

                if (isset($existing_schedules[$key])) {
                    ParlourListHasSchedule::where('id', $existing_schedules[$key])->update($scheduleData);
                    $processed_schedule_ids[] = $existing_schedules[$key];
                } else {
                    $scheduleData['parlour_list_id'] = $parlour_list->id;
                    $scheduleData['created_at'] = now();
                    $new_schedule = ParlourListHasSchedule::create($scheduleData);
                    $processed_schedule_ids[] = $new_schedule->id;
                }
            }

            $removed_schedules = array_diff($existing_schedules, $processed_schedule_ids);
            if (!empty($removed_schedules)) {
                ParlourListHasSchedule::whereIn('id', $removed_schedules)->delete();
            }

            $existing_services = $parlour_list->services()->pluck('id')->all();
            $processed_service_ids = [];

            foreach ($service_name as $key => $value) {
                $serviceData = [
                    'service_name' => $service_name[$key],
                    'price' => $price[$key],
                    'updated_at' => now(),
                ];


                if (isset($existing_services[$key])) {
                    ParlourHasService::where('id', $existing_services[$key])->update($serviceData);
                    $processed_service_ids[] = $existing_services[$key];
                } else {
                    $serviceData['parlour_list_id'] = $parlour_list->id;
                    $serviceData['created_at'] = now();
                    $new_service = ParlourHasService::create($serviceData);
                    $processed_service_ids[] = $new_service->id;
                }
            }

            $removed_services = array_diff($existing_services, $processed_service_ids);
            if (!empty($removed_services)) {
                ParlourHasService::whereIn('id', $removed_services)->delete();
            }

            $existing_stuff_ids = $parlour_list->stuff()->pluck('id')->toArray();
            $selected_stuff_ids = $stuff_name ?? [];
            $update_to_null = array_diff($existing_stuff_ids, $selected_stuff_ids);
            $update_to_parlour = array_intersect($existing_stuff_ids, $selected_stuff_ids);

            if (!empty($update_to_null)) {
                ParlourHasStuff::whereIn('id', $update_to_null)->update([
                    'Parlour_list_id' => null,
                    'status' => true,
                    'updated_at' => now(),
                ]);
            }

            if (!empty($update_to_parlour)) {
                ParlourHasStuff::whereIn('id', $update_to_parlour)->update([
                    'Parlour_list_id' => $parlour_list->id,
                    'status' => false,
                    'updated_at' => now(),
                ]);
            }
        } catch (Exception $e) {
            return back()->with(['error' => ['Something went wrong! Please try again.']]);
        }

        return redirect()->route('vendors.parlour.list.index')->with(['success' => ['Salon Updated Successfully!']]);
    }



    /**
     * Method for delete the parlour list information
     * @param \Illuminate\Http\Request $request
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'target'    => 'required|numeric',
        ]);

        $parlour_list   = ParlourList::find($request->target);
        $manager  = Manager::find($parlour_list->manager_id);
        try {
            $parlour_list->delete();
            $manager->delete();
        } catch (Exception $e) {
            return back()->with(['error'    =>  ['Something went wrong. Please try again!']]);
        }
        return back()->with(['success'  =>  ['Salon list deleted successfully.']]);
    }

    /**
     * Function for update admin status
     * @param  \Illuminate\Http\Request  $request
     */
    public function statusUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'data_target'       => "required|string|max:100",
            'status'            => "required|boolean",
        ]);

        if ($validator->stopOnFirstFailure()->fails()) {
            $error = ['error' => $validator->errors()];
            return back()->with(['error'    =>  ['Something went wrong. Please try again!']]);
        }

        $validated = $validator->safe()->all();
        $id = $validated['data_target'];

        $parlour = ParlourList::where('id', $id)->first();
        if (!$parlour) {
            $error = ['error' => [__('Salon not found!')]];
            return Response::error($error, null, 404);
        }

        try {
            $parlour->update([
                'active_status' => ($validated['status'] == true) ? false : true,
            ]);
        } catch (Exception $e) {
            $error = ['error' => [__('Something went wrong!. Please try again.')]];
            return Response::error($error, null, 500);
        }

        $success = ['success' => [__('Salon status updated successfully!')]];
        return Response::success($success, null, 200);
    }

    /**
     * Method for image validate
     * @param string $slug
     * @param \Illuminate\Http\Request  $request
     */
    public function imageValidate($request, $input_name, $old_image = null)
    {
        if ($request->hasFile($input_name)) {
            $image_validated = Validator::make($request->only($input_name), [
                $input_name         => "image|mimes:png,jpg,webp,jpeg,svg",
            ])->validate();

            $image = get_files_from_fileholder($request, $input_name);
            $upload = upload_files_from_path_dynamic($image, 'site-section', $old_image);
            return $upload;
        }
        return false;
    }

    public function deleteImage(Request $request)
    {
        // Validate the request
        $request->validate([
            'image_id' => 'required'
        ]);

        try {
            // Find and delete the image
            $image = ParlourImage::findOrFail($request->image_id);

            $image->delete();

            // Return success response
            return response()->json([
                'success' => true,
                'message' => 'Image deleted successfully'
            ]);
        } catch (\Exception $e) {
            // Return error response
            return response()->json([
                'success' => false,
                'message' => 'Error deleting image: ' . $e->getMessage()
            ], 500);
        }
    }
}
